# Define paths
$scriptDir = Split-Path -Parent $PSCommandPath
$configPath = Join-Path $scriptDir 'config.json'
$lastCheckPath = Join-Path $scriptDir 'last-check.txt'
$logDir = Join-Path $scriptDir 'logs'
$logFileBase = "upload-results.log"
$logFilePath = Join-Path $logDir $logFileBase

# Ensure PowerShell 7+ is being used
if ($PSVersionTable.PSVersion.Major -lt 7) {
    $msg = "ERROR: This script requires PowerShell 7 or later. You are using PowerShell $($PSVersionTable.PSVersion). See https://aka.ms/PSWindows"
    Write-Error $msg
    exit 1
}

# Load config
if (-Not (Test-Path $configPath)) {
    Write-Error "ERROR: Config file not found at $configPath"
    exit 1
}

try {
    $config = Get-Content $configPath | ConvertFrom-Json
    $watchFolder  = $config.watchFolder
    $mapping      = $config.mapping
    $apiKey       = $config.apiKey
    $url          = $config.url
    if (-not $url) { $url = "https://api.labhqlims.com/v1/upload-results" }
    $maxLogSizeMB = $config.maxLogSizeMB
    $maxLogFiles  = $config.maxLogFiles
    if (-not $maxLogSizeMB) { $maxLogSizeMB = 1 }
    if (-not $maxLogFiles)  { $maxLogFiles  = 5 }
} catch {
    Write-Error "ERROR: Failed to read or parse config.json - $_"
    exit 1
}

if (-Not (Test-Path $watchFolder)) {
    Write-Error "ERROR: Watched folder does not exist: $watchFolder"
    exit 1
}

if (-not $apiKey) {
    Write-Error "ERROR: Missing 'apiKey' in config.json"
    exit 1
}

if (-not $mapping) {
    Write-Error "ERROR: Missing 'mapping' in config.json"
    exit 1
}

# Ensure log directory exists
if (-not (Test-Path $logDir)) {
    New-Item -Path $logDir -ItemType Directory | Out-Null
}

# Rotate logs if necessary
function Rotate-Logs {
    if (-not (Test-Path $logFilePath)) { return }

    $maxBytes = $maxLogSizeMB * 1MB
    $fileSize = (Get-Item $logFilePath).Length

    if ($fileSize -lt $maxBytes) { return }

    # Rotate down (log.4 → log.5, ..., log → log.1)
    for ($i = $maxLogFiles - 1; $i -ge 1; $i--) {
        $old = Join-Path $logDir "$logFileBase.$i"
        $new = Join-Path $logDir ("$logFileBase." + ($i + 1))
        if (Test-Path $old) {
            Move-Item -Path $old -Destination $new -Force
        }
    }

    Move-Item -Path $logFilePath -Destination "$logFilePath.1" -Force
}

# Logging function
function Log {
    param ($message)
    Rotate-Logs
    $timestamp = (Get-Date).ToUniversalTime().ToString("o")
    Add-Content -Path $logFilePath -Value "$timestamp`t$message"
}

# Load last check time (in UTC)
if (Test-Path $lastCheckPath) {
    try {
        $lastCheckStr = Get-Content $lastCheckPath
        $lastCheck = [datetime]::Parse($lastCheckStr).ToUniversalTime()
    } catch {
        Log "WARNING: Invalid timestamp in last-check.txt, defaulting to epoch"
        $lastCheck = [datetime]::Parse("1970-01-01T00:00:00Z")
    }
} else {
    $lastCheck = [datetime]::Parse("1970-01-01T00:00:00Z")
}

# Get new or modified files since last check
$newFiles = Get-ChildItem -Path $watchFolder -File -Recurse | Where-Object {
    $_.LastWriteTimeUtc -gt $lastCheck
}

if ($newFiles.Count -eq 0) {
    Log "INFO: No new or modified files found."
} else {
    foreach ($file in $newFiles) {
        try {
            Log "INFO: Uploading: $($file.FullName)"

            $form = @{
                file    = Get-Item $file.FullName
                mapping = $mapping
            }

            $headers = @{
                Authorization = "Bearer $apiKey"
            }

            Invoke-RestMethod -Uri $url `
                              -Method Post `
                              -Form $form `
                              -Headers $headers

            Log "SUCCESS: Uploaded $($file.Name)"
        } catch {
            Log "ERROR: Failed to upload $($file.FullName) - $_"
        }
    }
}

# Save current UTC time as the last check
(Get-Date).ToUniversalTime().ToString("o") | Set-Content $lastCheckPath
