# Define paths
$scriptDir = Split-Path -Parent $PSCommandPath
$configPath = Join-Path $scriptDir 'config.json'
$scriptPath = Join-Path $scriptDir 'upload-results.ps1'
# ProgramData prevents the script from being overwritten
$installPath = Join-Path "$Env:ProgramData" 'Broughton Software'
$installScriptPath = Join-Path $installPath 'upload-results.ps1'
# This will mean that installing the script from any location will override previous installs,
# as the name is fixed. It's still probably the right move.
# This needs to be kept in sync with the uninstall script
$taskName = "Broughton Software LabHQ Results Uploader"

function Exit-WithPrompt {
    Write-Host "Press Enter to exit..."
    [void][System.Console]::ReadLine()
    exit 1
}

$pwsh = Get-Command pwsh.exe -ErrorAction SilentlyContinue
if (-not $pwsh) {
    $msg = "ERROR: The upload script requires PowerShell 7 or later. To install, see https://aka.ms/PSWindows"
    Write-Error $msg
    Exit-WithPrompt
}

# Validate script existence
if (-not (Test-Path $scriptPath)) {
    Write-Error "ERROR: upload-results.ps1 not found at $scriptPath"
    Exit-WithPrompt
}

# If we cannot find the config, the script will never run successfully.
# We may want to also validate the config here, but the user could change it afterwards.
if (-not (Test-Path $configPath)) {
    Write-Error "ERROR: config.json not found at $configPath"
    Exit-WithPrompt
}

try {
    $config = Get-Content $configPath | ConvertFrom-Json
    $intervalMinutes = $config.intervalMinutes
    if (-not $intervalMinutes) { $intervalMinutes = 30 }
} catch {
    Write-Error "ERROR: Failed to parse config.json - $_"
    Exit-WithPrompt
}

# Relaunch script with elevation if not run as Administrator
if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()
    ).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {

    Write-Host "WARNING: Not running as administrator. Requesting elevation..."

    $psArgs = "-NoExit -NoProfile -ExecutionPolicy Bypass -File `"$PSCommandPath`""
    Start-Process -FilePath "pwsh.exe" -ArgumentList $psArgs -Verb RunAs
    exit
}

# Ensure desired install directory exists
if (-not (Test-Path $installPath)) {
    New-Item -Path $installPath -ItemType Directory | Out-Null
}

# Copy the script + config to install directory
Copy-Item $configPath -Destination $installPath
Copy-Item $scriptPath -Destination $installPath

# Give users write permissions on $installPath; deny users write permissions on script
$BsAcl = Get-Acl -Path $installPath
$allowRuleParams = @{
  TypeName = 'System.Security.AccessControl.FileSystemAccessRule'
  # 3 = inherit to files and folders, 0 = propagate to all children
  ArgumentList = "BUILTIN\Users", "Write", 3, 0, "Allow"
}
$allowRule = New-Object @allowRuleParams
$BsAcl.SetAccessRule($allowRule)
Set-Acl -Path $installPath -AclObject $BsAcl

$scriptAcl = Get-Acl -Path $installScriptPath
$denyRuleParams= @{
  TypeName = 'System.Security.AccessControl.FileSystemAccessRule'
  ArgumentList = "BUILTIN\Users", "Write", "Deny"
}
$denyRule = New-Object @denyRuleParams
$scriptAcl.SetAccessRule($denyRule)
Set-Acl -Path $installScriptPath -AclObject $scriptAcl

$action = New-ScheduledTaskAction -Execute "pwsh.exe" -Argument "-NoProfile -ExecutionPolicy Bypass -File `"$installScriptPath`""
# ideally this gets run shortly after the process dumping files has completed. We have no idea
# how that would be scheduled, though, so "every 30 minutes starting now" is reasonable
$trigger = New-ScheduledTaskTrigger -Once -At (Get-Date).AddMinutes(1) -RepetitionInterval (New-TimeSpan -Minutes $intervalMinutes)
# SYSTEM user has far more permissions than necessary...
# but we don't know what account the user wants. Any local account we don't know the password to
# we can't make the task run when the user isn't logged in.
$principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Limited

# Register the task, overwriting any previously installed task
try {
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    Register-ScheduledTask -TaskName $taskName -Action $action -Trigger $trigger -Principal $principal -ErrorAction Stop
    Write-Output "INFO: Task '$taskName' registered to run every $intervalMinutes minutes."
} catch {
    Write-Error "ERROR: Failed to register scheduled task: $_"
    Exit-WithPrompt
}
